/***************************** Include Files *********************************/
#include "xparameters.h"
#include "xgpio.h"
#include "xscugic.h"
#include "xil_exception.h"
/************************** Constant Definitions *****************************/
/*
 * The following constant maps to the name of the hardware instances that
 * were created in the EDK XPS system.
 */
#define GPIO_EXAMPLE_DEVICE_ID XPAR_AXI_GPIO_0_DEVICE_ID
/*
 * The following constant is used to determine which channel of the GPIO is
 * used for the LED if there are 2 channels supported.
 */
#define LED_CHANNEL 1
#define BUTTON_CHANNEL 2
/************************** Variable Definitions *****************************/
/*
 * The following are declared globally so they are zeroed and so they are
 * easily accessible from a debugger
 */

#define INTC_DEVICE_ID XPAR_PS7_SCUGIC_0_DEVICE_ID
#define INTC_GPIO_INTERRUPT_ID 91
//#define INTC_GPIO_INTERRUPT_ID XPAR_FABRIC_GPIO_0_VEC_ID
#define INTC XScuGic
#define INTC_HANDLER XScuGic_InterruptHandler
#define BUTTON_INTERRUPT XGPIO_IR_CH2_MASK

XGpio Gpio; /* The Instance of the GPIO Driver */
static INTC Intc; /* The Instance of the Interrupt Controller Driver */
volatile u32 brightness;


/************************** Function Prototypes ******************************/

void GpioIsr(void *InstancePtr);
int SetupInterruptSystem();

/*****************************************************************************/
int main(void) {
	int Status;
	u32 value = 0;
	u32 period = 0;
	//u32 brightness = 0;

	/*
	 * Initialize the GPIO driver
	 */
	Status = XGpio_Initialize(&Gpio, GPIO_EXAMPLE_DEVICE_ID);
	if (Status != XST_SUCCESS) {
		return XST_FAILURE;
	}

	Status = SetupInterruptSystem();
	if (Status != XST_SUCCESS) {
		return XST_FAILURE;
	}
	// Clear LEDs
	XGpio_DiscreteWrite(&Gpio, LED_CHANNEL, 0);

	while (1) {
		print("Select a Brightness between 0 and 9\n\r");
		value = inbyte();
		period = value - 0x30;
		xil_printf("Brightness Level %d selected\n\r", period);
		// Since the LED width is 1e6 clk cycles, we need to normalize
		// the period to that clk.  Since we accept values 0-9, that will
		// scale period from 0-999,000.  0 turns off LEDs, 999,000 is full
		// brightness
		brightness = period * 110000;

		// Write the duty_cycle width (Period) out to the PL GPIO peripheral
		XGpio_DiscreteWrite(&Gpio, LED_CHANNEL, brightness);

	}

	return XST_SUCCESS;
	/*
	 * Setup the interrupts such that interrupt processing can occur.
	 * If an error occurs then exit
	 */

}

void GpioIsr(void *InstancePtr) {
	XGpio *GpioPtr = (XGpio *) InstancePtr;
	u32 Buttons; // Paste near top of function, with variables
	u32 New_DutyCycle; // Paste near top of function, with variables

	/*
	 * Disable the interrupt
	 */
	XGpio_InterruptDisable(GpioPtr, BUTTON_INTERRUPT);

	/*
	 * There should not be any other interrupts occurring other
	 * than the button changes
	 */
	if ((XGpio_InterruptGetStatus(GpioPtr) & BUTTON_INTERRUPT)
			!= BUTTON_INTERRUPT) {
		return;
	}

	/*
	 * Read state of push buttons and determine which ones changed
	 * states from the previous interrupt. Save a copy of the buttons
	 * for the next interrupt
	 */
	Buttons = XGpio_DiscreteRead(&Gpio, BUTTON_CHANNEL);

	if (Buttons == 1 && (brightness <= 880000)) {
		brightness = brightness + 110000;
	}
	/*
	 if (Buttons == 2 && (brightness <= 880000)) {
	 brightness = brightness + 110000;
	 }
	 if (Buttons == 4 && (brightness >= 110000)) {
	 brightness = brightness - 110000;
	 }
	 if (Buttons == 8 && (brightness >= 110000)) {
	 brightness = brightness - 110000;
	 }
	 if (Buttons == 16) {
	 brightness = 0;
	 */
	if (Buttons == 2 && (brightness >= 110000)) {
		brightness = brightness - 110000;
	}
	if (Buttons == 4) {
		brightness = 0;

	}
	New_DutyCycle = brightness / 110000;
	XGpio_DiscreteWrite(&Gpio, LED_CHANNEL, brightness);
	xil_printf("Brightness updated to: %d \n\r", New_DutyCycle);

	/* Clear the interrupt such that it is no longer pending in the GPIO */
	(void) XGpio_InterruptClear(GpioPtr, BUTTON_INTERRUPT);
	/*
	 * Enable the interrupt
	 */
	XGpio_InterruptEnable(GpioPtr, BUTTON_INTERRUPT);
	print("Interrupt Processed \r\n");
}

int SetupInterruptSystem() {
	int Result;
	INTC *IntcInstancePtr = &Intc;

#ifdef XPAR_INTC_0_DEVICE_ID
	/*
	 * Initialize the interrupt controller driver so that it's ready to use.
	 * specify the device ID that was generated in xparameters.h
	 */
	Result = XIntc_Initialize(IntcInstancePtr, INTC_DEVICE_ID);
	if (Result != XST_SUCCESS) {
		return Result;
	}

	/* Hook up interrupt service routine */
	XIntc_Connect(IntcInstancePtr, INTC_GPIO_INTERRUPT_ID,
			(Xil_ExceptionHandler)GpioIsr, &Gpio);

	/* Enable the interrupt vector at the interrupt controller */

	XIntc_Enable(IntcInstancePtr, INTC_GPIO_INTERRUPT_ID);

	/*
	 * Start the interrupt controller such that interrupts are recognized
	 * and handled by the processor
	 */
	Result = XIntc_Start(IntcInstancePtr, XIN_REAL_MODE);
	if (Result != XST_SUCCESS) {
		return Result;
	}

#else
	XScuGic_Config *IntcConfig;

	/*
	 * Initialize the interrupt controller driver so that it is ready to
	 * use.
	 */
	IntcConfig = XScuGic_LookupConfig(INTC_DEVICE_ID);
	if (NULL == IntcConfig) {
		return XST_FAILURE;
	}

	Result = XScuGic_CfgInitialize(IntcInstancePtr, IntcConfig,
			IntcConfig->CpuBaseAddress);
	if (Result != XST_SUCCESS) {
		return XST_FAILURE;
	}

	XScuGic_SetPriorityTriggerType(IntcInstancePtr, INTC_GPIO_INTERRUPT_ID,
			0xA0, 0x3);

	/*
	 * Connect the interrupt handler that will be called when an
	 * interrupt occurs for the device.
	 */
	Result = XScuGic_Connect(IntcInstancePtr, INTC_GPIO_INTERRUPT_ID,
			(Xil_ExceptionHandler) GpioIsr, &Gpio);
	if (Result != XST_SUCCESS) {
		return Result;
	}

	/*
	 * Enable the interrupt for the GPIO device.
	 */
	XScuGic_Enable(IntcInstancePtr, INTC_GPIO_INTERRUPT_ID);
#endif

	/*
	 * Enable the GPIO channel interrupts so that push button can be
	 * detected and enable interrupts for the GPIO device
	 */
	XGpio_InterruptEnable(&Gpio, BUTTON_INTERRUPT);
	XGpio_InterruptGlobalEnable(&Gpio);

	/*
	 * Initialize the exception table and register the interrupt
	 * controller handler with the exception table
	 */
	Xil_ExceptionInit();

	Xil_ExceptionRegisterHandler(XIL_EXCEPTION_ID_INT,
			(Xil_ExceptionHandler) INTC_HANDLER, IntcInstancePtr);

	/* Enable non-critical exceptions */Xil_ExceptionEnable();

	return XST_SUCCESS;
}

